/*******************************************************************************
  ProcessChecker Demo
  Example submitted by  by David Clegg

  This unit contains classes used to hold details for monitored and dependant
  processes.
*******************************************************************************/
using System;
using System.Collections;
using System.Runtime.Serialization;
using System.IO;

namespace ProcessChecker
{
	/// <summary>
	/// Base class to contain details of processes to monitor
	/// </summary>
	[Serializable]
	public class ProcessDetails {
		private string fProcessPath;

		/// <summary>
		/// The full path, including the executable name, for the process.
		/// </summary>
		public string ProcessPath {
			get {return fProcessPath;}
			set {fProcessPath = value;}
		}

		public ProcessDetails(string processPath) {
		  fProcessPath = processPath;
		}

		/// <summary>
		/// Name of the process to be used for calls to Process.GetProcesses.
		/// </summary>
		public string ProcessName {
			get {return Path.GetFileNameWithoutExtension(fProcessPath);}
		}
	}

	/// <summary>
	/// Class to represent a process that is to be watched, and restarted if
	/// necessary.
	/// </summary>
	[Serializable]
	public class WatchedProcess : ProcessDetails {

		private ProcessList fDependantProcesses;

		private bool fCheckResponding = true;

		public WatchedProcess(string processName) : base(processName) {
			fDependantProcesses = new ProcessList();
		}

		/// <summary>
		/// List of all dependant processes associated with this process.
		/// Dependant processes will be terminated before the watched process
		/// is restarted.
		/// </summary>
		public ProcessList DependantProcesses {
			get {return fDependantProcesses;}
		}

		/// <summary>
		/// Indicates whether ProcessChecker should check if the process is
		/// responding, and restart it if it isn't.
		/// </summary>
		public bool CheckResponding {
			get {return fCheckResponding;}
			set {fCheckResponding = value;}
		}
	}

	/// <summary>
	/// Class to represent a process that is a dependant of a watched process.
	/// If the watched process needs to be restarted, its dependants must be
	/// terminated first.
	/// </summary>
	[Serializable]
	public class DependantProcess : ProcessDetails {
		[NonSerialized]
		ProcessDetails fParent;
		public DependantProcess(string processName, ProcessDetails parent) : base(processName) {
			fParent = parent;
		}

		public ProcessDetails Parent {
			get {return fParent;}
		}
	}

	/// <summary>
	/// Strongly typed container class to hold references to multiple ProcessDetails
	/// classes.
	/// </summary>
	[Serializable]
	public class ProcessList : CollectionBase {
		public ProcessDetails this[int index] {
			get {return (ProcessDetails)List[index];}
			set {List[index] = value;}
		}

		public int Add(ProcessDetails process) {
			return List.Add(process);
		}

		public void Insert(int index, ProcessDetails process) {
			List.Insert(index, process);
		}

		public void Remove(ProcessDetails process) {
			List.Remove(process);
		}

		public bool Contains(ProcessDetails process) {
			return List.Contains(process);
		}

		public int IndexOf(ProcessDetails process) {
			return List.IndexOf(process);
		}
	}
}
